import os, random
from flask import Flask, make_response, request, flash, render_template, json, redirect, url_for, abort, session, g
from werkzeug.utils import secure_filename
from werkzeug.urls import url_parse

import uuid
from rq import Queue
from rq.job import Job
from worker import conn
import time
from datetime import datetime
from urllib.parse import urlparse
import sqlite3
from slugify import slugify

from dotenv import load_dotenv
load_dotenv()

from converters import image_converter, \
    audio_converter, video_converter, \
        document_converter, archive_converter, \
            ebook_converter, device_converter, webservice_converter, \
                pdf_converter, documents_compressor, image_compressor, \
                    video_compressor

from apscheduler.schedulers.background import BackgroundScheduler
from schedular import expire_files
from flask_babel import Babel, gettext as  _
from flask_minify import minify


sched = BackgroundScheduler(daemon=True)
sched.add_job(expire_files,'interval',minutes=60)
sched.start()

app = Flask(__name__)
app.secret_key = 'super secret key'

dir_path = os.path.dirname(os.path.realpath(__file__))

CONFIG_FOLDER = os.path.join(dir_path, "configs")

app.config['UPLOAD_DIR'] = os.path.join(dir_path, "static", "uploads")
app.config['BLOGS_ASSETS_DIR'] = os.path.join(dir_path, "static", "blogs")
app.config['BUCKET'] = os.getenv("aws_bucket")
app.config['LOCAL'] = False

if app.config['BUCKET'] == "" or app.config['BUCKET'] is None:
    app.config['LOCAL'] = True
    
app.jinja_env.trim_blocks = True
app.jinja_env.lstrip_blocks = True

babel = Babel(app)
minify(app=app, html=True, js=True, cssless=True)


q = Queue(connection=conn)

convert_list = {
    "image": image_converter,
    "bmp": image_converter,
    "audio": audio_converter,
    "video": video_converter,
    "document": document_converter,
    "archive": archive_converter,
    "ebook": ebook_converter,
    "device": device_converter,
    "webservice": webservice_converter,
    "pdf": pdf_converter,
    "document-compressor": documents_compressor,
    "image-compressor": image_compressor,
    "video-compressor": video_compressor
}

from configs.filetypes import available_filetypes
from configs.definition import definitions
from configs.languages import supported_languages
from configs.seo import seo
from configs.headings import headings

@babel.localeselector
def get_locale():
    return "en" if not g.lang_code \
        else g.lang_code

@app.route('/switch-language/<lang>')
def swith_language(lang):
    if lang in supported_languages.keys():
       g.lang_code = lang

    if request.referrer:
        path = replace_url_lang(request.referrer, lang)
        return redirect(
            path
        )
    else:
        return redirect("/")

def replace_url_lang(url, lang, full=False):
    path = url_parse(url).path
        
    paths = path.split("/")
    paths.pop(0)

    if len(paths) > 0 and paths[0] in supported_languages.keys():
        prev = "/" + paths[0]
        next = "/" + lang

        path = path.replace(prev, next if lang != "en" else "")
    
    elif lang != "en":
        path = "/" + lang + path
    
    return path if not full else request.url_root.rstrip("/") + path

@app.url_defaults
def add_language_code(endpoint, values):
    if 'lang_code' in values or not g.lang_code:
        return
    if app.url_map.is_endpoint_expecting(endpoint, 'lang_code'):
        values['lang_code'] = g.lang_code

@app.url_value_preprocessor
def pull_lang_code(endpoint, values):
    if values is not None and 'lang_code' in values:
        lang_code = values.pop('lang_code', None)

        if lang_code in supported_languages.keys() and lang_code != "en":
            g.lang_code = lang_code
        else:
            abort(404)
    else:
        g.lang_code = "en"

@app.route("/<lang_code>")
@app.route("/<lang_code>/")
@app.route("/")
def index():
    return render_template('index.html', filetypes = available_filetypes)


@app.route('/converter/<filetype>', methods = ['GET'])
@app.route("/<lang_code>/converter/<filetype>", methods = ['GET'])
@app.route('/converter/<filetype>/<fileformat>', methods = ['GET', 'POST'])
@app.route('/<lang_code>/converter/<filetype>/<fileformat>', methods = ['GET', 'POST'])
@app.route('/<first>-to-<second>', methods = ['GET', 'POST'])
@app.route('/<lang_code>/<first>-to-<second>', methods = ['GET', 'POST'])
def converter(filetype=None, fileformat=None, first=None, second=None):
    if fileformat == None and first == None and second == None:
        return render_template('list.html', 
                filetypes = available_filetypes,
                filetype = filetype,
                fileformat = fileformat
                )

    if (first is not None and second is not None):
        for k, i in available_filetypes.items():
            if first.upper() in map(str.upper, i['allowed']) and second.upper() in map(str.upper, i['ext']):
                filetype = k
                fileformat = second.upper()
                break


    if request.method == "GET":
        return render_template('converter.html', 
                filetypes = available_filetypes,
                filetype = filetype,
                fileformat = fileformat,
                options= json.dumps(available_filetypes[filetype]['options'], sort_keys = False, indent = 2)
                )
    else:
        urls = request.form.getlist('url[]')
        if len(urls) == 0:
            flash(_("Please upload a file to convert."), "error")
            return redirect(
                url("/converter/{}/{}".format(filetype, fileformat))
            )
        if len(urls) >= 10:
            flash(_("Sorry! currently we only support 10 files batch."), "error")
            return redirect(
                url("/converter/{}/{}".format(filetype, fileformat))
            )


        job = q.enqueue_call(
                func=convert_list[filetype].convert, 
                args=(urls, 
                        fileformat,
                        json.loads(request.form.get('options')),
                        {
                            "UPLOAD_DIR": app.config['UPLOAD_DIR'],
                            "BUCKET": app.config['BUCKET'],
                            "LOCAL": app.config['LOCAL']
                        }
                        ), 
                result_ttl=5000,
                timeout=500
            )

        return redirect(
            url("/output/{}".format(job.get_id()))
        )

@app.route('/<lang_code>/output/<job_id>', methods = ['GET'])                
@app.route('/output/<job_id>', methods = ['GET'])
def output(job_id):
    try:
        job = Job.fetch(job_id, connection=conn)
        # job = type('Job', (), {
        #     'is_finished': True,
        #     'result': type('Result', (), {
        #         'message': 'Success',
        #         'results': ["file.jpg"],
        #         'error': None
        #     })
        # })
    except:
        abort(404)
        return

    return render_template('output.html', job = job, job_id= job_id)


@app.route('/job/status/<job_id>', methods = ['GET'])
def jobStatus(job_id):
    try:
        job = Job.fetch(job_id, connection=conn)
        response = app.response_class(
            response = json.dumps({
                'is_finished': job.is_finished,
                'is_failed': job.is_failed,
                'is_queued': job.is_queued,
                'is_started': job.is_started
            }),
            status=200,
            mimetype='application/json'
        )
        return response
    except:
        abort(404)

@app.route('/<lang_code>/formats')
@app.route('/formats')
def formats():
    types = {}
    q = request.args.get("q")
    if q == None:
        q = ""
        
    for filetype in available_filetypes:
        for ext in available_filetypes[filetype]['ext']:
            if q != None and \
                (
                    q.lower() in ext.lower() or 
                    q.lower() in definitions[ext]
                ):
                types[ext] = {
                    "type": filetype,
                    "def" : definitions[ext] 
                }
            elif(q == None):
                types[ext] = {
                    "type": filetype,
                    "def" : definitions[ext] 
                } 
    return render_template('formats.html', types = types, q = q)

@app.route('/upload', methods = ['POST'])
def upload():
    files = request.files.getlist('files')
    filenames = []
    for file in files:
        filename = secure_filename(file.filename)
        n, e = os.path.splitext(filename)

        filename = n + e.lower()

        folder =  uuid.uuid4().hex

        os.makedirs(os.path.join(app.config['UPLOAD_DIR'], folder), exist_ok=True)
        file.save(os.path.join(app.config['UPLOAD_DIR'], folder , filename))

        filenames.append({
            "path" : os.path.join(folder, filename),
            "name" : n,
            "ext"  : e.lower()
        })

    response = app.response_class(
        response=json.dumps({
            'success': True,
            'files': filenames
        }),
        status=200,
        mimetype='application/json'
    )
    return response

@app.route('/upload/remove', methods = ['GET'])
def remove_folder():
    path = request.args.get('fileid')
    if path and len(path) != 0 \
        and os.path.exists(os.path.join(app.config['UPLOAD_DIR'], path)):
        os.remove(os.path.join(app.config['UPLOAD_DIR'], path))
    return "ok"

@app.route('/<lang_code>/blogs')
@app.route('/blogs')
def blogs():
    blogs = []
    with sqlite3.connect("database/sqlite.db") as connection:
        cursor = connection.cursor()

        cursor.execute("select rowid, * from blogs order by rowid desc")
        blogs = cursor.fetchall()

        cursor.close()
    return render_template("blogs.html", blogs=blogs)

@app.route('/<lang_code>/blogs/<slug>')
@app.route('/blogs/<slug>')
def blog(slug):
    print(slug)
    with sqlite3.connect("database/sqlite.db") as connection:
        cursor = connection.cursor()

        cursor.execute("select rowid, * from blogs where slug = ?", (slug, ))
        blog = cursor.fetchone()

        cursor.close()
        return render_template("blog.html", blog=blog)

    return redirect(
        url("/blogs")
    )

@app.route('/<lang_code>/admin')
@app.route('/admin')
def admin_home():
    return redirect(
        url("/admin/blogs")
    )

@app.route('/<lang_code>/admin/login', methods=["GET", "POST"])
@app.route('/admin/login', methods=["GET", "POST"])
def login():
    if (request.method == "GET"):
        error = session['error'] if "error" in session else None

        session.pop('error', None)
        return render_template("admin/login.html", error = error)
    else:
        username = request.form.get("username")
        password = request.form.get("password")

        if username == os.getenv("admin_username") and password == os.getenv("admin_password"):
            session['login'] = True
            print(g.lang_code)
            return redirect(
                url("/admin")
            )
        else:
            session['error'] = _("username or password is wrong")
            return redirect(
                url("/admin/login")
            )

@app.route('/<lang_code>/admin/blogs')
@app.route('/admin/blogs')
def admin_blogs():
    logged_in, r = authenticate()
    if not logged_in:
        return r

    with sqlite3.connect("database/sqlite.db") as connection:
        cursor = connection.cursor()
        cursor.execute("select rowid, * from blogs")

        blogs = cursor.fetchall()

        cursor.close()
    return render_template("admin/blogs.html", blogs=blogs)

@app.route('/<lang_code>/admin/blogs/<id>', methods=['GET', 'POST'])
@app.route('/admin/blogs/<id>', methods=['GET', 'POST'])
def admin_blogs_new(id):
    logged_in, r = authenticate()
    if not logged_in:
        return r

    if request.method == "GET":
        if id != "new":
            with sqlite3.connect("database/sqlite.db") as connection:
                cursor = connection.cursor()
                cursor.execute("select rowid, * from blogs where rowid = ?", (id,))

                blog = cursor.fetchone()
                cursor.close()

                return render_template("admin/blog.html", blog=blog)
        else:
            return render_template("admin/blog.html")
    else:
        title = request.form.get("title").strip()
        summary = request.form.get("summary").strip()
        content = request.form.get("content").strip()
        image = request.files.get("image")
        filename = request.form.get("featured")
        slug = slugify(title).strip()

        if (image is not None and image.filename != ""):
            filename = secure_filename(image.filename)
            filepath = os.path.join(app.config['BLOGS_ASSETS_DIR'], filename)

            image.save(filepath)

        with sqlite3.connect("database/sqlite.db") as connection:
            cursor = connection.cursor()

            if id == "new":
                cursor.execute("insert into blogs(title, summary, content, image, slug) values (?, ?, ? , ?, ?)", (title, summary, content, filename.strip(), slug))
            else:
                cursor.execute("update blogs set title = ?, summary = ?, content = ?, image = ?, slug = ? where rowid = ?", (title, summary, content, filename.strip(), slug, id))

            cursor.close()
            connection.commit()
            return { "status": "OK" }, 200
        
@app.route('/admin/blogs/remove/<id>', methods=['GET'])
def admin_blogs_remove(id):
    logged_in, r = authenticate()
    if not logged_in:
        return r

    with sqlite3.connect("database/sqlite.db") as connection:
        cursor = connection.cursor()

        cursor.execute("select * from blogs where rowid = ?", (id,))

        if not cursor.fetchone():
            cursor.close()
            return redirect(
                url("/admin/blogs")
            )
        else:
            cursor.execute("delete from blogs where rowid = ?", (id,))
            cursor.close()
            connection.commit()

        return redirect(
            redirect("/admin/blogs")
        )

@app.route('/<lang_code>/admin/logout')
@app.route('/admin/logout')
def admin_logout():
    session.pop('login', None)
    return redirect(
        url("/admin/login")
    )

@app.route("/<lang_code>/disclaimer")
@app.route('/disclaimer')
def disclaimer():
    return render_template("disclaimer.html")

@app.route("/<lang_code>/privacy")
@app.route('/privacy')
def privacy():
    return render_template("privacy.html")

@app.route("/<lang_code>/about")
@app.route('/about')
def about():
    return render_template("about.html")

@app.route("/<lang_code>/faq")
@app.route('/faq')
def faq():
    return render_template("faq.html")

@app.route("/<lang_code>/terms-of-use")
@app.route('/terms-of-use')
def terms():
    return render_template("terms-of-use.html")

@app.route("/<lang_code>/imprint")
@app.route('/imprint')
def imprint():
    return render_template("imprint.html")

@app.route("/<lang_code>/contact")
@app.route('/contact')
def contact():
    return render_template("contact.html")

@app.route("/<lang_code>/donate")
@app.route('/donate')
def donate():
    return render_template("donate.html")


@app.route("/sitemap")
@app.route("/sitemap/")
@app.route("/sitemap.xml")
def sitemap():
    host_components = urlparse(request.host_url)
    host_base = host_components.scheme + "://" + host_components.netloc

    # Static routes with static content
    static_urls = list()
    for rule in ["/", "/about", "/blogs", "/formats", "/faq", "/privacy", "/terms-of-use", "/imprint", "/contact", "/donate"]:
        url = {
            "loc": f"{host_base}{str(rule)}"
        }
        static_urls.append(url)
    
    for filetype, fileinfo in available_filetypes.items():
        url = {
                "loc": f"{host_base}/converter/{filetype}",
            }
        static_urls.append(url)
        for ext in fileinfo['ext']:
            url = {
                "loc": f"{host_base}/converter/{filetype}/{ext}",
            }
            static_urls.append(url)

    xml_sitemap = render_template("sitemap.xml", static_urls=static_urls, dynamic_urls=[], host_base=host_base)
    response = make_response(xml_sitemap)
    response.headers["Content-Type"] = "application/xml"

    return response

@app.template_filter('basename')
def basename(s):
    return os.path.basename(s)

@app.template_filter('splitpart')
def splitpart (value, index, char = ','):
    return value.split(char)[index]

def url(path):
    if g.lang_code == "en":
        return path
    return "/" + g.lang_code + path

def static_url(url):
    if not url.startswith("http"):
        url = url.replace(dir_path, "")
        if "/static/" not in url:
            url = "/static/uploads/" + url
    return url

def authenticate():
    if 'login' in session:
        return True, None
    else:
        request_xhr_key = request.headers.get('X-Requested-With')
        if request_xhr_key and request_xhr_key == 'XMLHttpRequest':
            return False, abort(403)
        else:
            return False, redirect(
                url("/admin/login")
            )

@app.context_processor
def functions():
    def get_definitions():
        for key in list(definitions.keys()):
            definitions[key.upper()] = definitions[key]
            definitions[key.lower()] = definitions[key]
        return definitions

    def get_types():
        return available_filetypes
    
    def get_converters():
        return [{**t, "key": _} for _, t in available_filetypes.items() if 'category' not in t]
    
    def get_tools(type):
        return [{**t, "key": _} for _, t in available_filetypes.items() if 'category' in t and _ == type]

    def ext_supported(key): 
        return key in [ext for k, i in available_filetypes.items() for ext in i['allowed'] ]

    def ext_color(counter): 
        if counter % 5 == 0:
            return "blue"
        elif counter % 3 == 0:
            return "green"
        elif counter % 2 == 0:
            return "red"
        else:
            return "yellow"
        
    def get_top_blogs():
        blogs = []
        with sqlite3.connect("database/sqlite.db") as connection:
            cursor = connection.cursor()

            cursor.execute("select rowid, * from blogs order by rowid desc limit 3")
            blogs = cursor.fetchall()

            cursor.close()
        return blogs
    
    def get_top_formats():
        types = {}
        for filetype in available_filetypes:
            for ext in available_filetypes[filetype]['ext']:
                types[ext] = {
                        "type": filetype,
                        "def" : definitions[ext] 
                    }
        keys = random.sample(types.keys(), 3)
        return {key: types[key] for key in keys}

    return {
        "definitions": get_definitions,
        "available_types": get_types,
        "config": app.config,
        "url_for": url_for,
        "url" : url,
        "join": os.path.join,
        "supported_languages": supported_languages,
        "replace_url_lang": replace_url_lang,
        "current_url": request.url,
        "dir_path": dir_path,
        "static_url": static_url,
        "ext_supported": ext_supported,
        "ext_color": ext_color,
        "seo": seo,
        "headings": headings,
        "get_converters": get_converters,
        "get_tools": get_tools,
        "get_top_blogs": get_top_blogs,
        "get_top_formats": get_top_formats
    }

if __name__ == "__main__":
    app.run(debug=True)